using StardewModdingAPI;
using StardewModdingAPI.Events;
using StardewValley;
using System;
using System.Collections.Generic;
using System.Linq;

namespace LevelUpTogether
{
    public class ModEntry : Mod
    {
        public static IMonitor SMonitor;
        public static IModHelper SHelper;
        public static ModConfig Config;

        private readonly Dictionary<long, PlayerSkillData> playerSkillData = new();

        public override void Entry(IModHelper helper)
        {
            Config = Helper.ReadConfig<ModConfig>();
            SMonitor = Monitor;
            SHelper = helper;

            helper.Events.GameLoop.DayEnding += OnDayEnding;
            helper.Events.GameLoop.SaveLoaded += OnSaveLoaded;
            helper.Events.Multiplayer.ModMessageReceived += OnModMessageReceived;
            helper.Events.GameLoop.GameLaunched += OnGameLaunched;
        }

        private void OnGameLaunched(object sender, GameLaunchedEventArgs e)
        {
            var configMenu = Helper.ModRegistry.GetApi<IGenericModConfigMenuApi>("spacechase0.GenericModConfigMenu");
            if (configMenu is null)
                return;

            configMenu.Register(
                mod: ModManifest,
                reset: () => Config = new ModConfig(),
                save: () => Helper.WriteConfig(Config)
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Enable Mod",
                getValue: () => Config.ModEnabled,
                setValue: value => Config.ModEnabled = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Farming",
                getValue: () => Config.SyncFarming,
                setValue: value => Config.SyncFarming = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Mining",
                getValue: () => Config.SyncMining,
                setValue: value => Config.SyncMining = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Foraging",
                getValue: () => Config.SyncForaging,
                setValue: value => Config.SyncForaging = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Fishing",
                getValue: () => Config.SyncFishing,
                setValue: value => Config.SyncFishing = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Combat",
                getValue: () => Config.SyncCombat,
                setValue: value => Config.SyncCombat = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Sync Luck",
                getValue: () => Config.SyncLuck,
                setValue: value => Config.SyncLuck = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Enable in Single Player",
                getValue: () => Config.EnableInSinglePlayer,
                setValue: value => Config.EnableInSinglePlayer = value
            );

            configMenu.AddBoolOption(
                mod: ModManifest,
                name: () => "Show Sync Messages",
                getValue: () => Config.ShowSyncMessages,
                setValue: value => Config.ShowSyncMessages = value
            );
        }

        private void OnSaveLoaded(object sender, SaveLoadedEventArgs e)
        {
            playerSkillData.Clear();
        }

        private void OnDayEnding(object sender, DayEndingEventArgs e)
        {
            try
            {
                if (!ShouldProcessSync())
                    return;

                if (Context.IsMainPlayer)
                {
                    CollectAndSyncPlayerData();
                }
            }
            catch (Exception ex)
            {
                Monitor.Log($"Error in OnDayEnding: {ex}", LogLevel.Error);
            }
        }

        private bool ShouldProcessSync()
        {
            if (!Config.ModEnabled)
                return false;

            if (!Context.IsWorldReady)
                return false;

            if (Game1.player == null)
                return false;

            if (!Context.IsMultiplayer && !Config.EnableInSinglePlayer)
                return false;

            return true;
        }

        private void CollectAndSyncPlayerData()
        {
            var allPlayers = Game1.getAllFarmers().ToList();
            var skillMaxData = new Dictionary<int, SkillLevelData>();

            foreach (var player in allPlayers)
            {
                var playerData = GetPlayerSkillData(player);
                
                foreach (var skill in playerData.Skills)
                {
                    if (!skillMaxData.ContainsKey(skill.Key) || 
                        skill.Value.ExperiencePoints > skillMaxData[skill.Key].ExperiencePoints)
                    {
                        skillMaxData[skill.Key] = skill.Value;
                    }
                }
            }

            ApplySkillSync(skillMaxData);
            BroadcastSkillSync(skillMaxData);
        }

        private PlayerSkillData GetPlayerSkillData(Farmer player)
        {
            var skills = new Dictionary<int, SkillLevelData>();

            if (Config.SyncFarming)
                skills[Farmer.farmingSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.farmingSkill],
                    Level = player.FarmingLevel 
                };

            if (Config.SyncMining)
                skills[Farmer.miningSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.miningSkill],
                    Level = player.MiningLevel 
                };

            if (Config.SyncForaging)
                skills[Farmer.foragingSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.foragingSkill],
                    Level = player.ForagingLevel 
                };

            if (Config.SyncFishing)
                skills[Farmer.fishingSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.fishingSkill],
                    Level = player.FishingLevel 
                };

            if (Config.SyncCombat)
                skills[Farmer.combatSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.combatSkill],
                    Level = player.CombatLevel 
                };

            if (Config.SyncLuck)
                skills[Farmer.luckSkill] = new SkillLevelData 
                { 
                    ExperiencePoints = player.experiencePoints[Farmer.luckSkill],
                    Level = player.LuckLevel 
                };

            return new PlayerSkillData
            {
                PlayerId = player.UniqueMultiplayerID,
                PlayerName = player.Name,
                Skills = skills
            };
        }

        private void ApplySkillSync(Dictionary<int, SkillLevelData> maxSkillData)
        {
            var player = Game1.player;
            var changedSkills = new List<string>();

            foreach (var skillData in maxSkillData)
            {
                int skillIndex = skillData.Key;
                var targetData = skillData.Value;
                int currentExp = player.experiencePoints[skillIndex];

                if (currentExp < targetData.ExperiencePoints)
                {
                    player.experiencePoints[skillIndex] = targetData.ExperiencePoints;
                    changedSkills.Add(GetSkillName(skillIndex));
                }
            }

            if (changedSkills.Any() && Config.ShowSyncMessages)
            {
                string message = $"Skills synchronized: {string.Join(", ", changedSkills)}";
                Game1.addHUDMessage(new HUDMessage(message, HUDMessage.newQuest_type));
            }
        }

        private void BroadcastSkillSync(Dictionary<int, SkillLevelData> maxSkillData)
        {
            if (!Context.IsMultiplayer)
                return;

            var syncData = new SkillSyncData
            {
                MaxSkillData = maxSkillData
            };

            Helper.Multiplayer.SendMessage(syncData, "SkillSync", new[] { ModManifest.UniqueID });
        }

        private void OnModMessageReceived(object sender, ModMessageReceivedEventArgs e)
        {
            try
            {
                if (e.FromModID != ModManifest.UniqueID)
                    return;

                if (e.Type == "SkillSync")
                {
                    var syncData = e.ReadAs<SkillSyncData>();
                    ApplySkillSync(syncData.MaxSkillData);
                }
            }
            catch (Exception ex)
            {
                Monitor.Log($"Error processing mod message: {ex}", LogLevel.Error);
            }
        }

        private string GetSkillName(int skillIndex)
        {
            return skillIndex switch
            {
                Farmer.farmingSkill => "Farming",
                Farmer.miningSkill => "Mining",
                Farmer.foragingSkill => "Foraging", 
                Farmer.fishingSkill => "Fishing",
                Farmer.combatSkill => "Combat",
                Farmer.luckSkill => "Luck",
                _ => "Unknown"
            };
        }
    }

    public class PlayerSkillData
    {
        public long PlayerId { get; set; }
        public string PlayerName { get; set; }
        public Dictionary<int, SkillLevelData> Skills { get; set; } = new();
    }

    public class SkillLevelData
    {
        public int ExperiencePoints { get; set; }
        public int Level { get; set; }
    }

    public class SkillSyncData
    {
        public Dictionary<int, SkillLevelData> MaxSkillData { get; set; } = new();
    }

    public interface IGenericModConfigMenuApi
    {
        void Register(IManifest mod, Action reset, Action save, bool titleScreenOnly = false);
        void AddBoolOption(IManifest mod, Func<bool> getValue, Action<bool> setValue, Func<string> name, Func<string> tooltip = null, string fieldId = null);
    }
}